<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Branch;
use App\Models\Supplier;
use App\Models\Inventory;
use App\Helpers\Restructure;
use App\Models\Summary;
use App\Models\Intake;

class IntakesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $intakes = Intake::query();

        if ($request->date) {
            $intakes->whereDate('date',$request->date);
        }

        if ($request->supplier_id) {

            if (str_starts_with($request->supplier_id, "B")) {
                $intakes->where('branch_id',str_replace("B","",$request->supplier_id));
            }else{
                $intakes->where('supplier_id',str_replace("S","",$request->supplier_id));
            }
        }

        if ($request->branch_id) {
            $intakes->whereHas('summary', function($query) use($request){
                $query->where('branch_id',$request->branch_id);
            });
        }

        if ($request->inventory_id) {
            $intakes->where('inventory_id',$request->inventory_id);
        }

        $branches = Branch::all()->map(function($branch) {
            return [
                'id' => "B".$branch->id,
                'name' => $branch->name,
            ];
        });

        $suppliers = Supplier::all()->map(function($supplier) {
            return [
                'id' => "S".$supplier->id,
                'name' => $supplier->name,
            ];
        });

        $sources = $branches->merge($suppliers);

        $intakes = $intakes->get();
        $branches = Branch::all();
        $inventories = Inventory::get();
        return view('intakes.index',compact('intakes','suppliers','branches','sources','inventories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $summary = Summary::find($request->summary_id);
        $branches = Branch::where('id','!=',$summary->branch_id)->get()->map(function($branch) {
            return [
                'id' => "B".$branch->id,
                'name' => $branch->name,
                // 'location' => $branch->location,
                // 'type' => 'Branch'
            ];
        });

        $suppliers = Supplier::all()->map(function($supplier) {
            return [
                'id' => "S".$supplier->id,
                'name' => $supplier->name,
                // 'location' => $supplier->location,
                // 'type' => 'Supplier'
            ];
        });

        $summary_intakes  = Intake::where('summary_id',$summary->id)->get();

        $intakes = [];
        foreach ($summary_intakes as $intake) {
            array_push($intakes, [
                'id' => ($intake->supplier_id) ? "S".$intake->supplier_id:"B".$intake->branch_id,
                'name'=>($intake->supplier_id) ? $intake->supplier?->name:$intake->branch?->name,
                'inventory_id' => $intake->inventory_id,
                'rate' => $intake->rate,
                'quantity' => $intake->quantity,
                'supplier_rate' => $intake->supplier_rate,
                'amount' => $intake->amount
            ]);
        }

        if (gettype($intakes) == "array") 
        {
            $intakes = json_encode($intakes);
        }

        $intakes = json_decode($intakes);

        $sources = $branches->merge($suppliers);

        $inventories = Inventory::all();

        return view('intakes.create',compact('sources','inventories','summary','intakes'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $summary = Summary::find($request->summary_id);

        try{
            $total = 0;

            \DB::beginTransaction();

            Intake::where('summary_id',$summary->id)->delete();

            foreach($request->items as $item){
                if (gettype($item) == "array") 
                    {
                        $item = json_encode($item);
                    }

                $item = json_decode($item);
                $branch_id = null;
                $supplier_id = null;
                if (str_starts_with($item->source, "B")) {
                    $branch_id = str_replace("B","",$item->source);
                }else{
                    $supplier_id = str_replace("S","",$item->source);
                }

                $item->rate = str_replace(",", "", $item->rate);
                $item->quantity = str_replace(",", "", $item->quantity);
                $item->supplier_rate = str_replace(",", "", $item->supplier_rate);

                $intake = new Intake;
                $intake->summary_id = $summary->id;
                $intake->date = $summary->date;
                $intake->supplier_id = $supplier_id;
                $intake->branch_id = $branch_id;
                $intake->inventory_id = $item->inventory_id;
                $intake->rate = $item->rate;
                $intake->supplier_rate = $item->supplier_rate;
                $intake->quantity = $item->quantity;
                $intake->amount = $item->quantity * $item->rate;
                $intake->created_by_id = \Auth::user()->id;
                $intake->save();

                $total += $intake->amount;
            }

            $summary->intake = $total;
            $summary->save();

            \DB::commit();

            return redirect()->back()->with('success','Intakes have been saved!');
        }catch(\Throwable $ex){
            \DB::rollback();
            return redirect()->back()->withErrors(['error'=>$ex->getMessage()])->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $intake = Intake::find($id);

        $branches = Branch::all()->map(function($branch) {
            return [
                'id' => "B".$branch->id,
                'name' => $branch->name,
            ];
        });

        $suppliers = Supplier::all()->map(function($supplier) {
            return [
                'id' => "S".$supplier->id,
                'name' => $supplier->name,
            ];
        });

        $sources = $branches->merge($suppliers);

        if ($intake->supplier_id) {
            $selected = "S".$intake->supplier_id;
        }else{
            $selected = "B".$intake->branch_id;
        }

        $inventories = Inventory::all();

        return view('intakes.edit',compact('intake','sources','selected','inventories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'source' => 'required',
            'rate' => 'required',
            'supplier_rate' => 'required',
            'quantity' => 'required',
            'inventory_id' => 'required',
        ]);

        try{
            \DB::beginTransaction();
            $branch_id = null;
            $supplier_id = null;
            if (str_starts_with($request->source, "B")) {
                $branch_id = str_replace("B","",$request->source);
            }else{
                $supplier_id = str_replace("S","",$request->source);
            }

            $request->rate = str_replace(",", "", $request->rate);
            $request->quantity = str_replace(",", "", $request->quantity);
            $request->supplier_rate = str_replace(",", "", $request->supplier_rate);

            $intake = Intake::find($id);
            $intake->date = $intake->summary->date;
            $intake->supplier_id = $supplier_id;
            $intake->branch_id = $branch_id;
            $intake->inventory_id = $request->inventory_id;
            $intake->rate = $request->rate;
            $intake->supplier_rate = $request->supplier_rate;
            $intake->quantity = $request->quantity;
            $intake->amount = $request->quantity * $request->rate;
            $intake->created_by_id = \Auth::user()->id;
            $intake->save();

            if ($supplier_id) {
                $restructure = new Restructure;
                $restructure->supplier_balances($supplier_id,null);
            }

            \DB::commit();

            return redirect()->back()->with('success','Update complete!');

        }catch(\Throwable $ex){
            \DB::rollback();
            return redirect()->back()->withErrors(['error'=>$ex->getMessage()])->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
