<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Disbursement;
use App\Models\Personel;
use App\Models\Ledger;
use App\Models\Payment;
use App\Models\Supplier;
use App\Models\DaySummary;
use App\Models\Summary;
use Yajra\DataTables\Facades\DataTables;
use Carbon\Carbon;

class DisbursementsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $disbursements = Disbursement::query();

            if ($request->date) {
                $disbursements->whereDate('date', $request->date);
            }

            if ($request->personel_id) {
                $disbursements->where('personel_id', $request->personel_id);
            }

            if ($request->supplier_id) {
                $credits->where('branch_id', $request->branch_id);
            }

            $disbursements->orderBy('orderNo','asc');

            return DataTables::of($disbursements)
                ->addColumn('personel', function ($row) {
                    return $row->personel->name ?? '-';
                })
                ->addColumn('supplier', function ($row) {
                    return $row->supplier->name ?? '-';
                })
                ->addColumn('amount', function ($row) {
                    return number_format($row->amount,2);
                })
                ->addColumn('balance', function ($row) {
                    return number_format($row->balance,2);
                })
                ->addColumn('action', function ($row) {
                    if (\Auth::user()->can('edit disbursements') || ($row->status == "Pending")) {
                       return '<a href="'.route('disbursements.edit', $row->id).'" class="btn btn-sm btn-'.(($row->status == "Pending") ? 'primary' : 'danger').'">Edit</a>';
                    }
                    
                })
                ->editColumn('date', function ($row) {
                    return date('d-M-Y',strtotime($row->date));
                })
                ->make(true);
        }

        $personels = Personel::all();
        $suppliers = Supplier::all();
        return view('disbursements.index', compact('personels', 'suppliers'));
    
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $personels = Personel::all();
        $suppliers = Supplier::all();
        $date = $request->date;

        $disbursements = Summary::whereDate('date',$date)->where('status','Open')->first();

        if ($disbursements) {
            return redirect()->back()->with('error','Can not start disbursements when u still have open summary!');
        }

        $requestDate = Carbon::parse($request->date);
        $yesterday = Carbon::yesterday();

        if ($requestDate->greaterThan($yesterday)) {
            return redirect()->back()->with('error','Invalid date selected!');
        }

        $disbursements = Disbursement::where('date',$request->date)->get();

        // if (sizeof($disbursements) > 0) {
        //    return redirect()->back()->with('error','Date already has disbursements');
        // }
        return view('disbursements.create',compact('suppliers','personels','disbursements','date'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'date' => 'required',
        ]);
        try {
            \DB::beginTransaction();
            $disbursements = Disbursement::where('date',$request->date)->where('status','!=','Approved')->delete();

            $summary = DaySummary::where('date',$request->date)->first();

            if (!$summary) {
                $summary = new DaySummary;
                $summary->date = $request->date;
                $summary->collections = 0;
                $summary->save();
            }

            $balance = $summary->collections;
            $total = 0;
           

            foreach($request->items as $index => $item){
                if (gettype($item) == "array") 
                {
                    $item = json_encode($item);
                }

                $item = json_decode($item);

                if ($item->status == "Approved") {
                    continue;
                }

                $item->amount = str_replace(",", "", $item->amount);

                $balance -= $item->amount;

                $disbursement = new Disbursement;
                $disbursement->date = $request->date;
                $disbursement->supplier_id = $item->supplier_id;
                $disbursement->personel_id = $item->personel_id;
                $disbursement->amount = $item->amount;
                $disbursement->reason = $item->reason;
                $disbursement->orderNo = $index + 1;
                $disbursement->balance = $balance;
                $disbursement->user_id = \Auth::user()->id;
                $disbursement->status = "Pending";
                $disbursement->save();

                $total += $disbursement->amount;


                $payment = new Payment;
                $payment->supplier_id = $item->supplier_id;
                $payment->customer_id = null;
                $payment->amount = $item->amount;
                $payment->disbursement_id = $disbursement->id;
                $payment->status = "Pending";
                $payment->date = $request->date;

                $balance = 0;

                $last_ledger_entry = Ledger::where('supplier_id', $item->supplier_id)->orderBy('id', 'desc')->first();
                if ($last_ledger_entry) {
                    $balance = $last_ledger_entry->balance;
                }

                $new_balance = $balance - $payment->amount;
                $payment->balance = $new_balance;
                $payment->save();

                $new_ledger_entry = new Ledger;
                $new_ledger_entry->customer_id = null;
                $new_ledger_entry->supplier_id = $item->supplier_id;
                $new_ledger_entry->credit = $item->amount;
                $new_ledger_entry->label = "Disbursement";
                $new_ledger_entry->payment_id = $payment->id;
                $new_ledger_entry->balance = $new_balance; 
                $new_ledger_entry->date = $payment->date;
                $new_ledger_entry->save();

                $supplier = Supplier::find($item->supplier_id);
                $supplier->balance = $new_balance;
                $supplier->save();
            }

            $summary->disbursements = $total;
            $summary->save();
            \DB::commit();
            return redirect()->back()->with('success','Disbursements have been saved!');
        } catch (\Throwable $e) {
            return $e;
            \DB::rollBack();
            return redirect()->back()->withErrors(['error'=>$e->getMessage()])->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
