<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Helpers\Restructure;
use App\Models\Customer;
use App\Models\Ledger;
use Dompdf\Options;
use Dompdf\Dompdf;


class CustomerController extends Controller {

    public function __construct() {
		// $this->authorizeResource(Customer::class, 'customer');
	}

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request, ) {

        $customers = Customer::query();

		if (!!$request->trashed) {
			$customers->withTrashed();
		}

        if(!empty($request->search)) {
			$customers->where('name', 'like', '%' . $request->search . '%');
		}

        $customers = $customers->get();

        return view('customers.index', compact('customers'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create() {

        return view('customers.create', []);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, ) {

        $request->validate(["name" => "required", "contact" => "required", "balance" => "required"]);

        try {

            \DB::beginTransaction();

            $customer = new Customer();
            $customer->name = $request->name;
    		$customer->contact = $request->contact;
    		$customer->alt_contact = $request->alt_contact;
    		$customer->balance = $request->balance;
            $customer->save();


            if ($customer->balance) {
                $ledger = new Ledger;
                $ledger->label = "Opening balance";
                $ledger->date = date('Y-m-d');
                $ledger->customer_id = $customer->id;
                $ledger->debit = $customer->balance;
                $ledger->balance = $customer->balance;
                $ledger->save();
            }

            \DB::commit();

            return redirect()->route('customers.index', [])->with('success', __('Customer created successfully.'));
        } catch (\Throwable $e) {
            \DB::rollback();
            return redirect()->route('customers.create', [])->withInput($request->input())->withErrors(['error' => $e->getMessage()]);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param \App\Models\Customer $customer
     *
     * @return \Illuminate\Http\Response
     */
    public function show(Customer $customer,) {

        return view('customers.show', compact('customer'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param \App\Models\Customer $customer
     *
     * @return \Illuminate\Http\Response
     */
    public function edit(Customer $customer,) {

        return view('customers.edit', compact('customer'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Customer $customer,) {

        $request->validate(["name" => "required", "contact" => "required"]);

        try {
            \DB::beginTransaction();
            $request->balance = str_replace(",","",$request->balance);
            $customer->name = $request->name;
    		$customer->contact = $request->contact;
    		$customer->alt_contact = $request->alt_contact;
            $customer->save();


            if ($request->balance) {
                if (!\Auth::user()->can('edit customer opening balance')) {
                    \DB::rollback();
                    return redirect()->back()->withErrors(['error'=>'You do not have permission to edit customer opening balance'])->withInput();
                }
                $restructure = new Restructure;
                $restructure->customer_balances($customer->id,$request->balance);
            }

            \DB::commit();
            return redirect()->route('customers.show', $customer->id)->with('success', __('Customer edited successfully.'));
        } catch (\Throwable $e) {
            \DB::rollback();
            return redirect()->route('customers.edit', compact('customer'))->withInput($request->input())->withErrors(['error' => $e->getMessage()]);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param \App\Models\Customer $customer
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Customer $customer,) {

        try {
            if($customer->balance > 0){
                return redirect()->back()->with('error','Customer has pending balance');
            }
            $customer->delete();

            return redirect()->route('customers.index', [])->with('success', __('Customer deleted successfully'));
        } catch (\Throwable $e) {
            return redirect()->route('customers.index', [])->with('error', 'Cannot delete Customer: ' . $e->getMessage());
        }
    }

    //@softdelete

    /**
     * Restore the specified deleted resource from storage.
     *
     * @param \App\Models\Customer $customer
     *
     * @return \Illuminate\Http\Response
     */
    public function restore( int $customer_id,) {

        $customer = Customer::withTrashed()->find($customer_id);
        // $this->authorize('delete', [Customer::class, $customer]);

        if (!empty($customer)) {
            $customer->restore();
            return redirect()->route('customers.index', [])->with('success', __('Customer restored successfully'));
        } else {
            return redirect()->route('customers.index', [])->with('error', 'Customer not found');
        }
    }

    public function purge( int $customer_id,) {

        $customer = Customer::withTrashed()->find($customer_id);
        // $this->authorize('delete', [Customer::class, $customer]);

        if (!empty($customer)) {
            $customer->forceDelete();
            return redirect()->route('customers.index', [])->with('success', __('Customer purged successfully'));
        } else {
            return redirect()->route('customers.index', [])->with('error', __('Customer not found'));
        }
    }
    
    public function ledger(Request $request)
    {
        $entries = null;
        $customer_id = null;

        if ($request->customer_id) {
            $customer_id = $request->customer_id;
            // $entries = Ledger::where('customer_id',$customer_id)->orderBy('id','asc')->get();

            // foreach ($entries as $entry) {
            //     if ($entry->credit_id) {
            //         $entry->module = "Credit";
            //     }elseif ($entry->collection_id) {
            //         $entry->module = "Collection";
            //     }elseif ($entry->intake_id) {
            //         $entry->module = "Intake";
            //     }elseif($entry->payment_id){
            //         $entry->module = "Payment";
            //     }else{
            //         $entry->module = "Others";
            //     }
            // }

            $entries = Ledger::where('customer_id', $customer_id)
                ->where('status','!=','Rejected')
                ->orderBy('id', 'asc')
                ->get()
                ->map(function ($entry) {
                    // Define mapping for module assignment
                    $mapping = [
                        'credit_id' => 'Credit',
                        'collection_id' => 'Collection',
                        'intake_id' => 'Intake',
                        'payment_id' => 'Payment',
                    ];

                    foreach ($mapping as $key => $value) {
                        if ($entry->$key) {
                            $entry->module = $value;
                            break;
                        }
                    }

                    // Default to "Others" if no module is matched
                    $entry->module = $entry->module ?? "Others";

                    return $entry;
                });

            // Add `deletable` attribute based on duplicates
            $uniqueEntries = [];
            $entries = $entries->map(function ($entry) use (&$uniqueEntries) {
                // Generate a unique key for duplicate detection
                $uniqueKey = null;

                if ($entry->credit_id) {
                    $uniqueKey = "credit_{$entry->credit_id}";
                } elseif ($entry->collection_id) {
                    $uniqueKey = "collection_{$entry->collection_id}";
                } elseif ($entry->intake_id) {
                    $uniqueKey = "intake_{$entry->intake_id}";
                } elseif ($entry->payment_id) {
                    $uniqueKey = "payment_{$entry->payment_id}";
                }

                // Check if this unique key already exists
                if ($uniqueKey && isset($uniqueEntries[$uniqueKey])) {
                    $entry->deletable = true;
                } else {
                    $entry->deletable = false;
                    if ($uniqueKey) {
                        $uniqueEntries[$uniqueKey] = true;
                    }
                }

                return $entry;
            });

        }

        $customers = Customer::all();

        if ($request->export == "pdf") {
            $customer = Customer::find($request->customer_id);
            $options = new Options();
            $options->set('defaultFont', 'Courier');
            $options->set('isRemoteEnabled', true);
            $options->set('isHtml5ParserEnabled', true);
            $options->set('isPhpEnabled', true);

            $dompdf = new Dompdf($options);
                
            $dompdf->loadHtml(view("customers.download_statement", compact('entries','customer'))->render());

            $dompdf->render();
            $dompdf->stream('Statement.pdf');
        }

        return view('customers.ledgers',compact('customer_id','entries','customers'));
    }

    public function delete_ledger($id){
        $entry = Ledger::find($id);

        if ($entry) {
            \DB::beginTransaction();
            $entry->delete();
            $restructure = new Restructure;
            if ($entry->customer_id) {
                $restructure->customer_balances($entry->customer_id,null);
            }

            if ($entry->supplier_id) {
                $restructure->supplier_balances($entry->supplier_id,null);
            }

            \DB::commit();
            return redirect()->back()->with('success', 'Entry deleted successfully.');
        }

        return redirect()->back()->with('error', 'Entry not found.');
    }

    public function clean_ledger(Request $request){

        \DB::beginTransaction();
        // $customer = Customer::find($request->customer_id);
        $restructure = new Restructure;
        if ($request->customer_id) {  
            $restructure->customer_balances($request->customer_id,null);
        }

        if ($request->supplier_id) {  
            $restructure->supplier_balances($request->supplier_id,null);
        }
        \DB::commit();

        return redirect()->back()->with('success','Ledger has been cleaned up');
    }


    public function clean_customer_ledgers(){
        $customers = Customer::all();

        foreach ($customers as $customer) {
            \DB::beginTransaction();
            $restructure = new Restructure;
            $restructure->customer_balances($customer->id,null);
            \DB::commit();
        }
    }

    public function download($value='')
    {
        $options = new Options();
        $options->set('defaultFont', 'Courier');
        $options->set('isRemoteEnabled', true);
        $options->set('isHtml5ParserEnabled', true);
        $options->set('isPhpEnabled', true);

        $dompdf = new Dompdf($options);

        $suppliers = Customer::orderBy('balance','asc')->get();

        // return $suppliers;
            
        $dompdf->loadHtml(view("customers.download", compact('suppliers'))->render());

        // Render the HTML as PDF
        $dompdf->render();
        $dompdf->stream('Suppliers.pdf');
    }

    public function suspense_account_transfers(Request $request,$id){
        $request->validate([
            'amount' => 'required',
            'type' => 'required',
        ]);
        // return $request;

        try {
            \DB::beginTransaction();
            $customer = Customer::find($id);

            $request->amount = str_replace(',', '', $request->amount);

            $suspenseAccountBalance = 0;
            $suspenseAccountEntry = \App\Models\SuspenseAccount::where('customer_id',$customer->id)->orderBy('id','desc')->first();

            if ($suspenseAccountEntry) {
                $suspenseAccountBalance = $suspenseAccountEntry->balance;
            }

            $transfer = new \App\Models\SuspenseAccount;
            $transfer->type = $request->type;
            $transfer->amount = $request->amount;
            $transfer->user_id = \Auth::user()->id;
            $transfer->customer_id = $customer->id;

            if ($request->type == "In") {
                $balance = ($customer->balance * -1);
                if ($balance < $request->amount) {
                    \DB::rollback();
                    return redirect()->back()->withErrors(['error'=>'Customer balance is less than amount you want to transfer to suspense account']);
                }

                $transfer->balance = $request->amount + $suspenseAccountBalance;
                $customer->balance += $request->amount;
                $customer->suspense_account_balance += $request->amount;
            }else{

                if ($suspenseAccountEntry < $request->amount) {
                    \DB::rollback();
                    return redirect()->back()->withErrors(['error'=>'Customer suspense balance is less than amount you want to transfer to main account']);
                }

                $transfer->balance = $suspenseAccountBalance - $request->amount;
                $customer->balance -= $request->amount;
                $customer->suspense_account_balance -= $request->amount;
            }

            $customer->save();
            $transfer->save();

            $ledger = Ledger::where('customer_id',$customer->id)->orderBy('id','desc')->first();

            $balance = 0;
            if ($ledger) {
                $balance = $ledger->balance;
            }

            $ledgerEntry = new Ledger;
            $ledgerEntry->customer_id = $customer->id;
            $ledgerEntry->supplier_id = null;
            $ledgerEntry->date = date('Y-m-d');
            $ledgerEntry->credit = (($request->type == "Out") ? $request->amount : null);
            $ledgerEntry->debit = (($request->type == "In") ? $request->amount: null);
            $ledgerEntry->label = "Credit Transfer";
            $ledgerEntry->balance = $customer->balance;
            $ledgerEntry->suspense_account_id = $transfer->id;
            $ledgerEntry->status = "Approved";
            $ledgerEntry->save();

            \DB::commit();
            return redirect('customers/'.$customer->id);
        } catch (\Throwable $e) {
            return $e->getMessage().' - '.$e->getLine();
        }
    }
}
