<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Models\Summary;
use App\Models\Customer;
use App\Models\Credit;
use App\Models\Branch;
use App\Helpers\Restructure;
use Yajra\DataTables\Facades\DataTables;

class CreditsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $credits = Credit::query();

            if ($request->date) {
                $credits->whereDate('date', $request->date);
            }

            if ($request->customer_id) {
                $credits->where('customer_id', $request->customer_id);
            }

            if ($request->branch_id) {
                $credits->where('branch_id', $request->branch_id);
            }

            $credits->orderBy('id','desc');

            return DataTables::of($credits)
                ->editColumn('date', function ($credit) {
                    return \Carbon\Carbon::parse($credit->date)->format('d-m-Y');
                })
                ->addColumn('customer_name', function ($credit) {
                    return $credit->customer->name ?? '-';
                })
                ->addColumn('amount', function ($credit) {
                    return number_format($credit->amount,2);
                })
                ->addColumn('branch_name', function ($credit) {
                    return $credit->branch->name ?? '-';
                })
                ->addColumn('action', function ($row) {
                    if (\Auth::user()->can('edit credits') || ($row->summary->status == "Open")) {
                       return '<a href="'.route('credits.edit', $row->id).'" class="btn btn-sm btn-'.(($row->summary?->status == "Open") ? 'primary' : 'danger').'">Edit</a>';
                    }
                    
                })
                ->editColumn('date', function ($row) {
                    return date('d-M-Y',strtotime($row->date));
                })
                ->make(true);
        }

        $branches = Branch::all();
        $customers = Customer::all();
        return view('credits.index', compact('customers', 'branches'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        $summary = Summary::find($request->summary_id);

        $customers = Customer::all();
        $credits = Credit::where('summary_id',$summary->id)->get();

        return view('credits.create',compact('customers','summary','credits'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try{
            $summary = Summary::find($request->summary_id);

            $total = 0;

            \DB::beginTransaction();


            $credits = Credit::where('summary_id',$summary->id)->delete();

            foreach($request->items as $item){
                if (gettype($item) == "array") 
                {
                    $item = json_encode($item);
                }

                $item = json_decode($item);

                $item->amount = str_replace(",", "", $item->amount);
                $credit = new Credit;
                $credit->summary_id = $summary->id;
                $credit->date = $summary->date;
                $credit->branch_id = $summary->branch_id;
                $credit->customer_id = $item->customer_id;
                $credit->receipt_no = $item->receipt_no;
                $credit->amount = $item->amount;
                $credit->created_by_id = \Auth::user()->id;
                $credit->save();

                $total += $credit->amount;
            }

            $summary->credit_sales = $total;
            $summary->save();

            \DB::commit();

            return redirect()->back()->with('success','Credits have been saved successfully!');
        }catch(\Throwable $ex){
            \DB::rollback();
            // return $ex->getMessage()." on line ".$ex->getLine();
            return redirect()->back()->withErrors(['error'=>$ex->getMessage()])->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $credit = Credit::find($id);
        $customers = Customer::all();

        return view('credits.edit',compact('credit','customers'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $request->validate([
            'customer_id' => ['required'],
            'amount' => ['required'],
            'receipt_no' => ['required'],
        ]);

        try{
            \DB::beginTransaction();
            $credit = Credit::find($id); // Be sure to validate $id exists
            $amount = str_replace(",", "", $request->amount); // Removes commas from the amount input

            $creditBefore = Credit::find($id);
            // Update credit details
            $credit->receipt_no = $request->receipt_no;
            $credit->customer_id = $request->customer_id;
            $credit->amount = $amount;
            $credit->save();

            $restructure = new Restructure;

            if ($credit->summary->status == "Closed") {
                if (($creditBefore->customer_id != $credit->customer_id)) {
                    $restructure->customer_balances($creditBefore->customer_id);
                    $restructure->customer_balances($credit->customer_id);
                }

                if (($creditBefore->amount != $credit->amount)) {
                    $restructure->branch_balances($credit->branch_id,$credit->summary_id);
                }
            }

            \DB::commit();

            return redirect()->back()->with('success','Credit Details have been updated!');;
        }catch(\Throwable $ex){
            \DB::rollback();
            return redirect()->back()->withErrors(['error' => $ex->getMessage().' on line '.$ex->getLine()])->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
